import { create } from 'zustand';
import type { AIProvider, AIModel, ChatMessage, Settings, Project } from '../types';

interface AppState {
  // AI state
  providers: AIProvider[];
  models: AIModel[];
  selectedProvider: string | null;
  selectedModel: string | null;

  // Chat state
  chatMessages: ChatMessage[];
  isStreaming: boolean;

  // Settings state
  settings: Settings;

  // Project state
  currentProject: Project | null;
  projects: Project[];

  // Editor state
  currentFile: string;
  files: Record<string, string>;

  // UI state
  currentPage: 'chat' | 'editor' | 'preview' | 'settings';
  isSidebarOpen: boolean;

  // Actions
  setProviders: (providers: AIProvider[]) => void;
  setModels: (models: AIModel[]) => void;
  setSelectedProvider: (provider: string) => void;
  setSelectedModel: (model: string) => void;

  addChatMessage: (message: ChatMessage) => void;
  setChatMessages: (messages: ChatMessage[]) => void;
  clearChatMessages: () => void;
  setIsStreaming: (isStreaming: boolean) => void;

  setSettings: (settings: Settings) => void;
  updateSettings: (settings: Partial<Settings>) => void;

  setCurrentProject: (project: Project | null) => void;
  setProjects: (projects: Project[]) => void;
  updateProjectFiles: (files: Record<string, string>) => void;

  setCurrentFile: (filename: string) => void;
  setFileContent: (filename: string, content: string) => void;

  setCurrentPage: (page: 'chat' | 'editor' | 'preview' | 'settings') => void;
  toggleSidebar: () => void;
}

export const useStore = create<AppState>((set) => ({
  // Initial state
  providers: [],
  models: [],
  selectedProvider: null,
  selectedModel: null,

  chatMessages: [],
  isStreaming: false,

  settings: {
    apiKeys: {},
    defaultProvider: 'openai',
    defaultModel: null,
    temperature: 0.7,
    maxTokens: 2000
  },

  currentProject: null,
  projects: [],

  currentFile: 'index.html',
  files: {
    'index.html': `<!DOCTYPE html>
<html lang="uz">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mening Loyiham</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <h1>Salom Dunyo!</h1>
    <p>AI Builder Platform bilan yaratilgan.</p>
    <script src="script.js"></script>
</body>
</html>`,
    'style.css': `* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', sans-serif;
    padding: 20px;
    background: #f5f5f5;
}

h1 {
    color: #333;
    margin-bottom: 10px;
}

p {
    color: #666;
    line-height: 1.6;
}`,
    'script.js': `console.log('AI Builder Platform - Ishga tushdi!');

// Sizning kodingiz shu yerda
document.addEventListener('DOMContentLoaded', () => {
    console.log('Sahifa yuklandi');
});`
  },

  currentPage: 'chat',
  isSidebarOpen: true,

  // Actions
  setProviders: (providers) => set({ providers }),
  setModels: (models) => set({ models }),
  setSelectedProvider: (provider) => set({ selectedProvider: provider }),
  setSelectedModel: (model) => set({ selectedModel: model }),

  addChatMessage: (message) =>
    set((state) => ({ chatMessages: [...state.chatMessages, message] })),
  setChatMessages: (messages) => set({ chatMessages: messages }),
  clearChatMessages: () => set({ chatMessages: [] }),
  setIsStreaming: (isStreaming) => set({ isStreaming }),

  setSettings: (settings) => set({ settings }),
  updateSettings: (newSettings) =>
    set((state) => ({ settings: { ...state.settings, ...newSettings } })),

  setCurrentProject: (project) => set({ currentProject: project }),
  setProjects: (projects) => set({ projects }),
  updateProjectFiles: (files) =>
    set((state) => ({
      files,
      currentProject: state.currentProject
        ? { ...state.currentProject, files }
        : null
    })),

  setCurrentFile: (filename) => set({ currentFile: filename }),
  setFileContent: (filename, content) =>
    set((state) => ({
      files: { ...state.files, [filename]: content }
    })),

  setCurrentPage: (page) => set({ currentPage: page }),
  toggleSidebar: () => set((state) => ({ isSidebarOpen: !state.isSidebarOpen }))
}));
