import axios from 'axios';
import type {
  AIProvider,
  AIModel,
  ChatMessage,
  ChatOptions,
  Settings,
  Project,
  APIResponse
} from '../types';

const API_BASE_URL = import.meta.env.VITE_API_URL || 'http://localhost:5000/api';

const api = axios.create({
  baseURL: API_BASE_URL,
  headers: {
    'Content-Type': 'application/json'
  },
  timeout: 120000
});

/**
 * AI API
 */
export const aiAPI = {
  /**
   * Qo'llab-quvvatlanadigan providerlar ro'yxati
   */
  async getProviders(): Promise<AIProvider[]> {
    const response = await api.get<APIResponse<{ providers: AIProvider[] }>>('/ai/providers');
    return response.data.data?.providers || [];
  },

  /**
   * API connection test
   */
  async testConnection(provider: string, apiKey: string): Promise<APIResponse> {
    const response = await api.post('/ai/test', { provider, apiKey });
    return response.data;
  },

  /**
   * Modellar ro'yxati
   */
  async getModels(provider: string, apiKey?: string): Promise<AIModel[]> {
    const response = await api.post<APIResponse<{ models: AIModel[] }>>('/ai/models', {
      provider,
      apiKey
    });
    return response.data.data?.models || [];
  },

  /**
   * Chat (oddiy)
   */
  async chat(
    provider: string,
    messages: ChatMessage[],
    options?: ChatOptions
  ): Promise<APIResponse> {
    const response = await api.post('/ai/chat', {
      provider,
      messages,
      options
    });
    return response.data;
  },

  /**
   * Chat streaming
   */
  async chatStream(
    provider: string,
    messages: ChatMessage[],
    options: ChatOptions,
    onChunk: (content: string) => void
  ): Promise<void> {
    const response = await fetch(`${API_BASE_URL}/ai/chat/stream`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        provider,
        messages,
        options
      })
    });

    if (!response.ok) {
      throw new Error('Streaming xatosi');
    }

    const reader = response.body?.getReader();
    const decoder = new TextDecoder();

    if (!reader) {
      throw new Error('Stream reader mavjud emas');
    }

    while (true) {
      const { done, value } = await reader.read();

      if (done) break;

      const chunk = decoder.decode(value);
      const lines = chunk.split('\n').filter(line => line.trim() !== '');

      for (const line of lines) {
        if (line.startsWith('data: ')) {
          try {
            const data = JSON.parse(line.slice(6));

            if (data.error) {
              throw new Error(data.error);
            }

            if (data.done) {
              return;
            }

            if (data.content) {
              onChunk(data.content);
            }
          } catch (e) {
            console.error('JSON parse xatosi:', e);
          }
        }
      }
    }
  }
};

/**
 * Settings API
 */
export const settingsAPI = {
  /**
   * Sozlamalarni olish
   */
  async getSettings(): Promise<Settings> {
    const response = await api.get<APIResponse<{ settings: Settings }>>('/settings');
    return response.data.data?.settings || {
      apiKeys: {},
      defaultProvider: 'openai',
      defaultModel: null,
      temperature: 0.7,
      maxTokens: 2000
    };
  },

  /**
   * Sozlamalarni saqlash
   */
  async saveSettings(settings: Partial<Settings>): Promise<APIResponse> {
    const response = await api.post('/settings', settings);
    return response.data;
  },

  /**
   * API keyni yangilash
   */
  async updateApiKey(provider: string, apiKey: string): Promise<APIResponse> {
    const response = await api.post('/settings/api-key', { provider, apiKey });
    return response.data;
  },

  /**
   * Sozlamalarni reset qilish
   */
  async resetSettings(): Promise<APIResponse> {
    const response = await api.post('/settings/reset');
    return response.data;
  }
};

/**
 * Project API
 */
export const projectAPI = {
  /**
   * Barcha loyihalar
   */
  async getProjects(): Promise<Project[]> {
    const response = await api.get<APIResponse<{ projects: Project[] }>>('/projects');
    return response.data.data?.projects || [];
  },

  /**
   * Bitta loyiha
   */
  async getProject(id: string): Promise<Project | null> {
    try {
      const response = await api.get<APIResponse<{ project: Project }>>(`/projects/${id}`);
      return response.data.data?.project || null;
    } catch (error) {
      return null;
    }
  },

  /**
   * Loyihani saqlash
   */
  async saveProject(project: Partial<Project>): Promise<APIResponse> {
    const response = await api.post('/projects', project);
    return response.data;
  },

  /**
   * Loyihani o'chirish
   */
  async deleteProject(id: string): Promise<APIResponse> {
    const response = await api.delete(`/projects/${id}`);
    return response.data;
  },

  /**
   * Loyihani export qilish (ZIP)
   */
  getExportUrl(id: string): string {
    return `${API_BASE_URL}/projects/${id}/export`;
  },

  /**
   * Platformani export qilish (ZIP)
   */
  getPlatformExportUrl(): string {
    return `${API_BASE_URL}/export/platform`;
  }
};

/**
 * Chat history API
 */
export const chatAPI = {
  /**
   * Chat tarixini saqlash
   */
  async saveChatHistory(sessionId: string, messages: ChatMessage[]): Promise<APIResponse> {
    const response = await api.post('/chat/save', { sessionId, messages });
    return response.data;
  },

  /**
   * Chat tarixini olish
   */
  async getChatHistory(sessionId: string): Promise<ChatMessage[]> {
    const response = await api.get<APIResponse<{ messages: ChatMessage[] }>>(`/chat/${sessionId}`);
    return response.data.data?.messages || [];
  }
};

export default api;
