import React, { useEffect, useState } from 'react';
import { useStore } from '../store/useStore';
import { settingsAPI, aiAPI } from '../services/api';

export const SettingsPage: React.FC = () => {
  const { settings, setSettings, providers, setProviders } = useStore();

  const [apiKeys, setApiKeys] = useState<Record<string, string>>({});
  const [defaultProvider, setDefaultProvider] = useState('openai');
  const [temperature, setTemperature] = useState(0.7);
  const [maxTokens, setMaxTokens] = useState(2000);

  const [testResults, setTestResults] = useState<Record<string, any>>({});
  const [testing, setTesting] = useState<Record<string, boolean>>({});
  const [saving, setSaving] = useState(false);
  const [message, setMessage] = useState<{ type: 'success' | 'error'; text: string } | null>(null);

  useEffect(() => {
    loadSettings();
    loadProviders();
  }, []);

  const loadSettings = async () => {
    try {
      const data = await settingsAPI.getSettings();
      setSettings(data);
      setApiKeys(data.apiKeys || {});
      setDefaultProvider(data.defaultProvider);
      setTemperature(data.temperature);
      setMaxTokens(data.maxTokens);
    } catch (error) {
      console.error('Sozlamalarni yuklashda xato:', error);
    }
  };

  const loadProviders = async () => {
    try {
      const data = await aiAPI.getProviders();
      setProviders(data);
    } catch (error) {
      console.error('Providerlarni yuklashda xato:', error);
    }
  };

  const handleTestApiKey = async (providerId: string) => {
    const apiKey = apiKeys[providerId];

    if (!apiKey || apiKey === '********') {
      setMessage({ type: 'error', text: 'API key kiriting' });
      return;
    }

    setTesting({ ...testing, [providerId]: true });
    setTestResults({ ...testResults, [providerId]: null });

    try {
      const result = await aiAPI.testConnection(providerId, apiKey);

      setTestResults({
        ...testResults,
        [providerId]: result
      });

      if (result.success) {
        setMessage({ type: 'success', text: `${providerId} muvaffaqiyatli ulandi!` });
      } else {
        setMessage({ type: 'error', text: result.error || 'Ulanishda xato' });
      }
    } catch (error: any) {
      setMessage({ type: 'error', text: error.message });
    } finally {
      setTesting({ ...testing, [providerId]: false });
    }
  };

  const handleSaveSettings = async () => {
    setSaving(true);
    setMessage(null);

    try {
      const newSettings = {
        apiKeys,
        defaultProvider,
        temperature,
        maxTokens
      };

      const result = await settingsAPI.saveSettings(newSettings);

      if (result.success) {
        setSettings({ ...settings, ...newSettings });
        setMessage({ type: 'success', text: 'Sozlamalar saqlandi!' });
      } else {
        setMessage({ type: 'error', text: result.error || 'Saqlashda xato' });
      }
    } catch (error: any) {
      setMessage({ type: 'error', text: error.message });
    } finally {
      setSaving(false);
    }
  };

  const handleResetSettings = async () => {
    if (!confirm('Sozlamalarni standart holatga qaytarmoqchimisiz?')) {
      return;
    }

    try {
      const result = await settingsAPI.resetSettings();

      if (result.success) {
        loadSettings();
        setMessage({ type: 'success', text: 'Sozlamalar standart holatga qaytarildi' });
      } else {
        setMessage({ type: 'error', text: result.error || 'Reset qilishda xato' });
      }
    } catch (error: any) {
      setMessage({ type: 'error', text: error.message });
    }
  };

  return (
    <div className="settings-page">
      <div className="settings-container">
        <h2>Sozlamalar</h2>

        {message && (
          <div className={`message message-${message.type}`}>
            {message.text}
          </div>
        )}

        <section className="settings-section">
          <h3>AI Provider API Keylar</h3>
          <p className="section-description">
            Har bir AI provider uchun API key kiriting. Keylar shifrlangan holda saqlanadi.
          </p>

          {providers.map((provider) => (
            <div key={provider.id} className="api-key-group">
              <label>{provider.name}</label>

              <div className="input-group">
                <input
                  type="password"
                  placeholder={`${provider.name} API key`}
                  value={apiKeys[provider.id] || ''}
                  onChange={(e) =>
                    setApiKeys({ ...apiKeys, [provider.id]: e.target.value })
                  }
                  disabled={!provider.requiresApiKey}
                />

                <button
                  className="btn btn-small"
                  onClick={() => handleTestApiKey(provider.id)}
                  disabled={
                    !provider.requiresApiKey ||
                    !apiKeys[provider.id] ||
                    apiKeys[provider.id] === '********' ||
                    testing[provider.id]
                  }
                >
                  {testing[provider.id] ? 'Tekshirilmoqda...' : 'Test'}
                </button>
              </div>

              {testResults[provider.id] && (
                <div
                  className={`test-result ${
                    testResults[provider.id].success ? 'success' : 'error'
                  }`}
                >
                  {testResults[provider.id].success ? (
                    <>
                      ✓ Muvaffaqiyatli ulandi
                      {testResults[provider.id].message && (
                        <div>{testResults[provider.id].message}</div>
                      )}
                    </>
                  ) : (
                    <>
                      ✗ {testResults[provider.id].error || 'Ulanishda xato'}
                    </>
                  )}
                </div>
              )}

              {!provider.requiresApiKey && (
                <div className="info-text">
                  Bu provider API key talab qilmaydi (local)
                </div>
              )}
            </div>
          ))}
        </section>

        <section className="settings-section">
          <h3>Standart Sozlamalar</h3>

          <div className="form-group">
            <label>Standart Provider</label>
            <select
              value={defaultProvider}
              onChange={(e) => setDefaultProvider(e.target.value)}
            >
              {providers.map((provider) => (
                <option key={provider.id} value={provider.id}>
                  {provider.name}
                </option>
              ))}
            </select>
          </div>

          <div className="form-group">
            <label>Temperature: {temperature}</label>
            <input
              type="range"
              min="0"
              max="2"
              step="0.1"
              value={temperature}
              onChange={(e) => setTemperature(parseFloat(e.target.value))}
            />
            <p className="help-text">
              0 = aniq javoblar, 2 = kreativ javoblar
            </p>
          </div>

          <div className="form-group">
            <label>Max Tokens: {maxTokens}</label>
            <input
              type="number"
              min="100"
              max="8000"
              step="100"
              value={maxTokens}
              onChange={(e) => setMaxTokens(parseInt(e.target.value))}
            />
            <p className="help-text">
              Javob uzunligi cheklovi (100-8000)
            </p>
          </div>
        </section>

        <div className="settings-actions">
          <button
            className="btn btn-primary"
            onClick={handleSaveSettings}
            disabled={saving}
          >
            {saving ? 'Saqlanmoqda...' : 'Saqlash'}
          </button>

          <button
            className="btn btn-secondary"
            onClick={handleResetSettings}
          >
            Standartga Qaytarish
          </button>
        </div>

        <section className="settings-section">
          <h3>Ma'lumot</h3>
          <div className="info-panel">
            <p><strong>Platform versiyasi:</strong> 1.0.0</p>
            <p><strong>Backend API:</strong> http://localhost:5000</p>
            <p><strong>Qo'llab-quvvatlanadigan AI lar:</strong> {providers.length} ta</p>
          </div>
        </section>
      </div>
    </div>
  );
};
