import React, { useEffect, useRef, useState } from 'react';
import { useStore } from '../store/useStore';
import { projectAPI } from '../services/api';

export const PreviewPage: React.FC = () => {
  const { files, currentProject } = useStore();
  const iframeRef = useRef<HTMLIFrameElement>(null);
  const [consoleOutput, setConsoleOutput] = useState<string[]>([]);
  const [showConsole, setShowConsole] = useState(false);

  useEffect(() => {
    updatePreview();
  }, [files]);

  const updatePreview = () => {
    if (!iframeRef.current) return;

    const htmlContent = files['index.html'] || '<h1>index.html topilmadi</h1>';
    const cssContent = files['style.css'] || '';
    const jsContent = files['script.js'] || '';

    // Console.log ni qo'lga olish uchun wrapper
    const consoleWrapper = `
      <script>
        (function() {
          const originalLog = console.log;
          const originalError = console.error;
          const originalWarn = console.warn;

          window.parentConsole = [];

          console.log = function(...args) {
            window.parentConsole.push({ type: 'log', message: args.join(' ') });
            window.parent.postMessage({ type: 'console', data: window.parentConsole }, '*');
            originalLog.apply(console, args);
          };

          console.error = function(...args) {
            window.parentConsole.push({ type: 'error', message: args.join(' ') });
            window.parent.postMessage({ type: 'console', data: window.parentConsole }, '*');
            originalError.apply(console, args);
          };

          console.warn = function(...args) {
            window.parentConsole.push({ type: 'warn', message: args.join(' ') });
            window.parent.postMessage({ type: 'console', data: window.parentConsole }, '*');
            originalWarn.apply(console, args);
          };

          window.onerror = function(message, source, lineno, colno, error) {
            console.error('Xato: ' + message + ' (' + source + ':' + lineno + ')');
            return false;
          };
        })();
      </script>
    `;

    // HTML ni to'liq qilish
    let fullHtml = htmlContent;

    // Agar <style> tag bo'lmasa, CSS ni qo'shish
    if (cssContent && !htmlContent.includes('<style>')) {
      const styleTag = `<style>${cssContent}</style>`;
      if (htmlContent.includes('</head>')) {
        fullHtml = htmlContent.replace('</head>', `${styleTag}</head>`);
      } else if (htmlContent.includes('<head>')) {
        fullHtml = htmlContent.replace('<head>', `<head>${styleTag}`);
      } else {
        fullHtml = `<head>${styleTag}</head>${htmlContent}`;
      }
    }

    // Agar <script> tag bo'lmasa, JS ni qo'shish
    if (jsContent && !htmlContent.includes('<script>')) {
      const scriptTag = `<script>${jsContent}</script>`;
      if (fullHtml.includes('</body>')) {
        fullHtml = fullHtml.replace('</body>', `${scriptTag}</body>`);
      } else {
        fullHtml = `${fullHtml}${scriptTag}`;
      }
    }

    // Console wrapper qo'shish
    if (fullHtml.includes('</head>')) {
      fullHtml = fullHtml.replace('</head>', `${consoleWrapper}</head>`);
    } else {
      fullHtml = `<head>${consoleWrapper}</head>${fullHtml}`;
    }

    // Iframe ni yangilash
    const iframe = iframeRef.current;
    iframe.srcdoc = fullHtml;
  };

  // Console output qabul qilish
  useEffect(() => {
    const handleMessage = (event: MessageEvent) => {
      if (event.data.type === 'console') {
        const logs = event.data.data.map((item: any) =>
          `[${item.type}] ${item.message}`
        );
        setConsoleOutput(logs);
      }
    };

    window.addEventListener('message', handleMessage);
    return () => window.removeEventListener('message', handleMessage);
  }, []);

  const handleRefresh = () => {
    setConsoleOutput([]);
    updatePreview();
  };

  const handleExport = () => {
    if (!currentProject) {
      alert('Loyiha saqlanmagan. Avval loyihani saqlang.');
      return;
    }

    const exportUrl = projectAPI.getExportUrl(currentProject.id);
    window.open(exportUrl, '_blank');
  };

  const handleExportPlatform = () => {
    const exportUrl = projectAPI.getPlatformExportUrl();
    window.open(exportUrl, '_blank');
  };

  return (
    <div className="preview-page">
      <div className="preview-toolbar">
        <h3>Ko'rish</h3>
        <div className="toolbar-buttons">
          <button className="btn btn-small" onClick={handleRefresh}>
            🔄 Yangilash
          </button>
          <button
            className="btn btn-small"
            onClick={() => setShowConsole(!showConsole)}
          >
            {showConsole ? '📕' : '📗'} Console
          </button>
          <button className="btn btn-small" onClick={handleExport}>
            📦 Loyihani Yuklab Olish
          </button>
          <button className="btn btn-small btn-primary" onClick={handleExportPlatform}>
            🚀 Platformani Yuklab Olish
          </button>
        </div>
      </div>

      <div className="preview-container">
        <iframe
          ref={iframeRef}
          className="preview-iframe"
          title="Preview"
          sandbox="allow-scripts allow-same-origin allow-forms allow-modals"
        />
      </div>

      {showConsole && (
        <div className="console-panel">
          <div className="console-header">
            <h4>Console</h4>
            <button
              className="btn btn-small"
              onClick={() => setConsoleOutput([])}
            >
              Tozalash
            </button>
          </div>
          <div className="console-output">
            {consoleOutput.length === 0 ? (
              <div className="console-empty">Console bo'sh</div>
            ) : (
              consoleOutput.map((log, index) => (
                <div
                  key={index}
                  className={`console-line ${
                    log.includes('[error]') ? 'error' :
                    log.includes('[warn]') ? 'warn' : 'log'
                  }`}
                >
                  {log}
                </div>
              ))
            )}
          </div>
        </div>
      )}
    </div>
  );
};
