import React, { useState } from 'react';
import Editor from '@monaco-editor/react';
import { useStore } from '../store/useStore';

export const EditorPage: React.FC = () => {
  const { currentFile, files, setCurrentFile, setFileContent } = useStore();
  const [newFileName, setNewFileName] = useState('');
  const [showNewFileDialog, setShowNewFileDialog] = useState(false);

  const fileLanguages: Record<string, string> = {
    html: 'html',
    css: 'css',
    js: 'javascript',
    ts: 'typescript',
    json: 'json',
    md: 'markdown'
  };

  const getLanguage = (filename: string): string => {
    const ext = filename.split('.').pop()?.toLowerCase() || '';
    return fileLanguages[ext] || 'plaintext';
  };

  const handleEditorChange = (value: string | undefined) => {
    if (value !== undefined) {
      setFileContent(currentFile, value);
    }
  };

  const handleCreateFile = () => {
    if (!newFileName.trim()) return;

    const filename = newFileName.trim();

    if (files[filename]) {
      alert('Bu nomdagi fayl allaqachon mavjud!');
      return;
    }

    setFileContent(filename, '');
    setCurrentFile(filename);
    setNewFileName('');
    setShowNewFileDialog(false);
  };

  const handleDeleteFile = (filename: string) => {
    if (Object.keys(files).length <= 1) {
      alert('Kamida bitta fayl bo\'lishi kerak!');
      return;
    }

    if (!confirm(`${filename} faylini o'chirmoqchimisiz?`)) {
      return;
    }

    const newFiles = { ...files };
    delete newFiles[filename];

    useStore.setState({ files: newFiles });

    // Boshqa faylga o'tish
    if (currentFile === filename) {
      setCurrentFile(Object.keys(newFiles)[0]);
    }
  };

  return (
    <div className="editor-page">
      <div className="editor-sidebar">
        <div className="files-header">
          <h3>Fayllar</h3>
          <button
            className="btn btn-small"
            onClick={() => setShowNewFileDialog(true)}
            title="Yangi fayl"
          >
            +
          </button>
        </div>

        {showNewFileDialog && (
          <div className="new-file-dialog">
            <input
              type="text"
              value={newFileName}
              onChange={(e) => setNewFileName(e.target.value)}
              placeholder="fayl.html"
              onKeyPress={(e) => {
                if (e.key === 'Enter') handleCreateFile();
              }}
              autoFocus
            />
            <div className="dialog-buttons">
              <button className="btn btn-small btn-primary" onClick={handleCreateFile}>
                Yaratish
              </button>
              <button
                className="btn btn-small btn-secondary"
                onClick={() => {
                  setShowNewFileDialog(false);
                  setNewFileName('');
                }}
              >
                Bekor
              </button>
            </div>
          </div>
        )}

        <div className="files-list">
          {Object.keys(files).map((filename) => (
            <div
              key={filename}
              className={`file-item ${currentFile === filename ? 'active' : ''}`}
            >
              <div
                className="file-name"
                onClick={() => setCurrentFile(filename)}
              >
                {filename}
              </div>
              {Object.keys(files).length > 1 && (
                <button
                  className="btn-icon"
                  onClick={() => handleDeleteFile(filename)}
                  title="O'chirish"
                >
                  ✕
                </button>
              )}
            </div>
          ))}
        </div>

        <div className="editor-info">
          <h4>Ko'rsatmalar</h4>
          <ul>
            <li>Monaco Editor ishlatilgan</li>
            <li>Syntax highlighting</li>
            <li>IntelliSense</li>
            <li>Ctrl+S - saqlash</li>
          </ul>
        </div>
      </div>

      <div className="editor-main">
        <div className="editor-header">
          <h3>{currentFile}</h3>
          <span className="file-type">{getLanguage(currentFile)}</span>
        </div>

        <Editor
          height="100%"
          language={getLanguage(currentFile)}
          value={files[currentFile] || ''}
          onChange={handleEditorChange}
          theme="vs-dark"
          options={{
            fontSize: 14,
            fontFamily: "'Cascadia Code', 'Fira Code', 'Consolas', monospace",
            minimap: { enabled: true },
            lineNumbers: 'on',
            roundedSelection: true,
            scrollBeyondLastLine: false,
            automaticLayout: true,
            tabSize: 2,
            wordWrap: 'on',
            formatOnPaste: true,
            formatOnType: true
          }}
        />
      </div>
    </div>
  );
};
