import React, { useState, useRef, useEffect } from 'react';
import { useStore } from '../store/useStore';
import { aiAPI } from '../services/api';
import type { ChatMessage } from '../types';

export const ChatPage: React.FC = () => {
  const {
    chatMessages,
    addChatMessage,
    clearChatMessages,
    isStreaming,
    setIsStreaming,
    selectedProvider,
    selectedModel,
    providers,
    models,
    setProviders,
    setModels,
    setSelectedProvider,
    setSelectedModel,
    settings
  } = useStore();

  const [inputMessage, setInputMessage] = useState('');
  const [systemPrompt, setSystemPrompt] = useState('Siz foydali AI yordamchisiz. Kod yozishda yordam bering.');
  const [temperature, setTemperature] = useState(0.7);
  const [maxTokens, setMaxTokens] = useState(2000);
  const [error, setError] = useState<string | null>(null);

  const messagesEndRef = useRef<HTMLDivElement>(null);

  // Providerlarni yuklash
  useEffect(() => {
    loadProviders();
  }, []);

  // Modellarni yuklash
  useEffect(() => {
    if (selectedProvider) {
      loadModels(selectedProvider);
    }
  }, [selectedProvider]);

  // Oxirgi xabarga scroll qilish
  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  }, [chatMessages]);

  const loadProviders = async () => {
    try {
      const data = await aiAPI.getProviders();
      setProviders(data);

      // Default providerni tanlash
      if (data.length > 0 && !selectedProvider) {
        const defaultProvider = data.find(p => p.id === settings.defaultProvider) || data[0];
        setSelectedProvider(defaultProvider.id);
      }
    } catch (err) {
      setError('Providerlarni yuklashda xato');
    }
  };

  const loadModels = async (provider: string) => {
    try {
      const data = await aiAPI.getModels(provider);
      setModels(data);

      // Default modelni tanlash
      if (data.length > 0 && !selectedModel) {
        setSelectedModel(data[0].id);
      }
    } catch (err) {
      console.error('Modellarni yuklashda xato:', err);
    }
  };

  const handleSendMessage = async () => {
    if (!inputMessage.trim() || !selectedProvider) {
      return;
    }

    const userMessage: ChatMessage = {
      role: 'user',
      content: inputMessage.trim(),
      timestamp: Date.now()
    };

    addChatMessage(userMessage);
    setInputMessage('');
    setError(null);
    setIsStreaming(true);

    try {
      // Xabarlar ro'yxatini tayyorlash
      const messages: ChatMessage[] = [];

      if (systemPrompt.trim()) {
        messages.push({ role: 'system', content: systemPrompt.trim() });
      }

      messages.push(...chatMessages, userMessage);

      // Streaming javob
      let assistantContent = '';

      const assistantMessage: ChatMessage = {
        role: 'assistant',
        content: '',
        timestamp: Date.now()
      };

      addChatMessage(assistantMessage);

      await aiAPI.chatStream(
        selectedProvider,
        messages,
        {
          model: selectedModel || undefined,
          temperature,
          maxTokens,
          stream: true
        },
        (chunk) => {
          assistantContent += chunk;

          // Oxirgi xabarni yangilash
          useStore.setState((state) => ({
            chatMessages: state.chatMessages.map((msg, idx) =>
              idx === state.chatMessages.length - 1
                ? { ...msg, content: assistantContent }
                : msg
            )
          }));
        }
      );
    } catch (err: any) {
      setError(err.message || 'Xatolik yuz berdi');

      // Xato xabarini qo'shish
      addChatMessage({
        role: 'assistant',
        content: `❌ Xatolik: ${err.message || 'Noma\'lum xato'}`,
        timestamp: Date.now()
      });
    } finally {
      setIsStreaming(false);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  };

  return (
    <div className="chat-page">
      <div className="chat-sidebar">
        <h3>Sozlamalar</h3>

        <div className="form-group">
          <label>AI Provider</label>
          <select
            value={selectedProvider || ''}
            onChange={(e) => setSelectedProvider(e.target.value)}
            disabled={isStreaming}
          >
            <option value="">Tanlang...</option>
            {providers.map((provider) => (
              <option key={provider.id} value={provider.id}>
                {provider.name}
              </option>
            ))}
          </select>
        </div>

        <div className="form-group">
          <label>Model</label>
          <select
            value={selectedModel || ''}
            onChange={(e) => setSelectedModel(e.target.value)}
            disabled={isStreaming || !selectedProvider}
          >
            <option value="">Tanlang...</option>
            {models.map((model) => (
              <option key={model.id} value={model.id}>
                {model.name}
              </option>
            ))}
          </select>
        </div>

        <div className="form-group">
          <label>System Prompt</label>
          <textarea
            value={systemPrompt}
            onChange={(e) => setSystemPrompt(e.target.value)}
            rows={4}
            disabled={isStreaming}
            placeholder="System prompt kiriting..."
          />
        </div>

        <div className="form-group">
          <label>Temperature: {temperature}</label>
          <input
            type="range"
            min="0"
            max="2"
            step="0.1"
            value={temperature}
            onChange={(e) => setTemperature(parseFloat(e.target.value))}
            disabled={isStreaming}
          />
        </div>

        <div className="form-group">
          <label>Max Tokens: {maxTokens}</label>
          <input
            type="number"
            min="100"
            max="8000"
            step="100"
            value={maxTokens}
            onChange={(e) => setMaxTokens(parseInt(e.target.value))}
            disabled={isStreaming}
          />
        </div>

        <button
          className="btn btn-secondary"
          onClick={clearChatMessages}
          disabled={isStreaming || chatMessages.length === 0}
        >
          Chatni Tozalash
        </button>
      </div>

      <div className="chat-main">
        <div className="chat-messages">
          {chatMessages.length === 0 && (
            <div className="empty-state">
              <h2>AI Chat</h2>
              <p>Xabar yozing va AI bilan suhbatni boshlang</p>
            </div>
          )}

          {chatMessages.map((message, index) => (
            <div key={index} className={`message message-${message.role}`}>
              <div className="message-header">
                <strong>
                  {message.role === 'user'
                    ? 'Siz'
                    : message.role === 'assistant'
                    ? 'AI'
                    : 'Sistema'}
                </strong>
              </div>
              <div className="message-content">
                <pre>{message.content}</pre>
              </div>
            </div>
          ))}

          <div ref={messagesEndRef} />
        </div>

        {error && (
          <div className="error-message">
            ❌ {error}
          </div>
        )}

        <div className="chat-input-container">
          <textarea
            className="chat-input"
            value={inputMessage}
            onChange={(e) => setInputMessage(e.target.value)}
            onKeyPress={handleKeyPress}
            placeholder="Xabar yozing... (Enter - yuborish, Shift+Enter - yangi qator)"
            rows={3}
            disabled={isStreaming || !selectedProvider}
          />
          <button
            className="btn btn-primary"
            onClick={handleSendMessage}
            disabled={!inputMessage.trim() || isStreaming || !selectedProvider}
          >
            {isStreaming ? 'Yuklanmoqda...' : 'Yuborish'}
          </button>
        </div>
      </div>
    </div>
  );
};
