import {
  OpenAIAdapter,
  AnthropicAdapter,
  GoogleAdapter,
  OpenRouterAdapter,
  DeepSeekAdapter,
  OllamaAdapter
} from '../adapters/index.js';

/**
 * AI Orchestrator
 * Barcha AI providerlarni boshqaradi
 * Yangi AI qo'shish: adapters papkasida yangi adapter yaratib, bu yerga qo'shing
 */
export class AIOrchestrator {
  constructor() {
    this.adapters = new Map();
  }

  /**
   * Provider uchun adapter yaratish
   */
  getAdapter(provider, apiKey, config = {}) {
    // Agar allaqachon yaratilgan bo'lsa, qaytarish
    const cacheKey = `${provider}-${apiKey?.substring(0, 10)}`;
    if (this.adapters.has(cacheKey)) {
      return this.adapters.get(cacheKey);
    }

    let adapter;

    switch (provider.toLowerCase()) {
      case 'openai':
        adapter = new OpenAIAdapter(apiKey, config);
        break;

      case 'anthropic':
        adapter = new AnthropicAdapter(apiKey, config);
        break;

      case 'google':
        adapter = new GoogleAdapter(apiKey, config);
        break;

      case 'openrouter':
        adapter = new OpenRouterAdapter(apiKey, config);
        break;

      case 'deepseek':
        adapter = new DeepSeekAdapter(apiKey, config);
        break;

      case 'ollama':
        adapter = new OllamaAdapter(apiKey || 'local', config);
        break;

      default:
        throw new Error(`Noma'lum provider: ${provider}`);
    }

    this.adapters.set(cacheKey, adapter);
    return adapter;
  }

  /**
   * API ulanishini test qilish
   */
  async testConnection(provider, apiKey, config = {}) {
    try {
      const adapter = this.getAdapter(provider, apiKey, config);
      return await adapter.testConnection();
    } catch (error) {
      return {
        success: false,
        error: error.message
      };
    }
  }

  /**
   * Modellar ro'yxatini olish
   */
  async getModels(provider, apiKey, config = {}) {
    try {
      const adapter = this.getAdapter(provider, apiKey, config);
      return await adapter.getModels();
    } catch (error) {
      return {
        success: false,
        error: error.message,
        models: []
      };
    }
  }

  /**
   * Chat (oddiy)
   */
  async chat(provider, apiKey, messages, options = {}) {
    try {
      const adapter = this.getAdapter(provider, apiKey, options.config);

      // Input validatsiya
      if (!messages || !Array.isArray(messages) || messages.length === 0) {
        return {
          success: false,
          error: 'Xabarlar massivi bo\'sh'
        };
      }

      return await adapter.chat(messages, options);
    } catch (error) {
      console.error('Chat xatosi:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  /**
   * Chat streaming
   */
  async chatStream(provider, apiKey, messages, options = {}, onChunk) {
    try {
      const adapter = this.getAdapter(provider, apiKey, options.config);

      // Input validatsiya
      if (!messages || !Array.isArray(messages) || messages.length === 0) {
        throw new Error('Xabarlar massivi bo\'sh');
      }

      return await adapter.chatStream(messages, options, onChunk);
    } catch (error) {
      console.error('Chat streaming xatosi:', error);
      throw error;
    }
  }

  /**
   * Qo'llab-quvvatlanadigan providerlar ro'yxati
   */
  getSupportedProviders() {
    return [
      {
        id: 'openai',
        name: 'OpenAI',
        requiresApiKey: true,
        defaultModel: 'gpt-4-turbo-preview'
      },
      {
        id: 'anthropic',
        name: 'Anthropic (Claude)',
        requiresApiKey: true,
        defaultModel: 'claude-3-sonnet-20240229'
      },
      {
        id: 'google',
        name: 'Google Gemini',
        requiresApiKey: true,
        defaultModel: 'gemini-pro'
      },
      {
        id: 'openrouter',
        name: 'OpenRouter',
        requiresApiKey: true,
        defaultModel: 'openai/gpt-3.5-turbo'
      },
      {
        id: 'deepseek',
        name: 'DeepSeek',
        requiresApiKey: true,
        defaultModel: 'deepseek-chat'
      },
      {
        id: 'ollama',
        name: 'Ollama (Local)',
        requiresApiKey: false,
        defaultModel: 'llama2'
      }
    ];
  }

  /**
   * Cache ni tozalash
   */
  clearCache() {
    this.adapters.clear();
  }
}

// Singleton instance
export const aiOrchestrator = new AIOrchestrator();
