import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import dotenv from 'dotenv';
import routes from './routes/index.js';
import { initStorage } from './utils/storage.js';
import { validateJSON } from './middleware/validator.js';

// Environment variables yuklash
dotenv.config();

// Express app yaratish
const app = express();
const PORT = process.env.PORT || 5000;

/**
 * Middleware
 */

// Helmet - xavfsizlik headerlari
app.use(helmet({
  crossOriginResourcePolicy: { policy: 'cross-origin' }
}));

// CORS - barcha domainlarga ruxsat
app.use(cors({
  origin: '*',
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization']
}));

// Body parser
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// JSON parse xatolarini tutish
app.use(validateJSON);

/**
 * Storage ni init qilish
 */
initStorage();

/**
 * Routes
 */
app.use('/api', routes);

// Root endpoint
app.get('/', (req, res) => {
  res.json({
    success: true,
    message: 'AI Builder Platform API',
    version: '1.0.0',
    endpoints: {
      health: '/api/health',
      providers: '/api/ai/providers',
      settings: '/api/settings',
      projects: '/api/projects',
      chat: '/api/ai/chat',
      chatStream: '/api/ai/chat/stream'
    }
  });
});

/**
 * 404 handler
 */
app.use((req, res) => {
  res.status(404).json({
    success: false,
    error: 'Endpoint topilmadi'
  });
});

/**
 * Global error handler
 */
app.use((err, req, res, next) => {
  console.error('Server xatosi:', err);

  res.status(err.status || 500).json({
    success: false,
    error: err.message || 'Ichki server xatosi',
    ...(process.env.NODE_ENV === 'development' && { stack: err.stack })
  });
});

/**
 * Serverni ishga tushirish
 */
app.listen(PORT, () => {
  console.log('╔════════════════════════════════════════╗');
  console.log('║   AI Builder Platform Backend API      ║');
  console.log('╠════════════════════════════════════════╣');
  console.log(`║   Server: http://localhost:${PORT}       ║`);
  console.log(`║   Environment: ${process.env.NODE_ENV || 'development'}             ║`);
  console.log('║   Status: ✓ Ishga tushdi               ║');
  console.log('╚════════════════════════════════════════╝');
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal qabul qilindi. Server yopilmoqda...');
  process.exit(0);
});

process.on('SIGINT', () => {
  console.log('\nSIGINT signal qabul qilindi. Server yopilmoqda...');
  process.exit(0);
});

export default app;
