import express from 'express';
import {
  getSupportedProviders,
  testApiConnection,
  getModels,
  chat,
  chatStream
} from '../controllers/aiController.js';
import {
  getSettingsHandler,
  saveSettingsHandler,
  updateApiKey,
  resetSettings
} from '../controllers/settingsController.js';
import {
  getProjects,
  getProjectById,
  saveProjectHandler,
  deleteProjectHandler,
  exportProject,
  exportPlatform
} from '../controllers/projectController.js';
import {
  saveChatHistoryHandler,
  getChatHistoryHandler
} from '../controllers/chatController.js';
import {
  apiLimiter,
  aiLimiter,
  exportLimiter
} from '../middleware/rateLimiter.js';
import { validateRequestBody, validateBody } from '../middleware/validator.js';

const router = express.Router();

/**
 * AI endpointlar
 */
router.get('/ai/providers', getSupportedProviders);
router.post('/ai/test', validateRequestBody, validateBody(), aiLimiter, testApiConnection);
router.post('/ai/models', validateRequestBody, validateBody(), getModels);
router.post('/ai/chat', validateRequestBody, validateBody(), aiLimiter, chat);
router.post('/ai/chat/stream', validateRequestBody, validateBody(), aiLimiter, chatStream);

/**
 * Sozlamalar endpointlar
 */
router.get('/settings', getSettingsHandler);
router.post('/settings', validateRequestBody, validateBody(), saveSettingsHandler);
router.post('/settings/api-key', validateRequestBody, validateBody(), updateApiKey);
router.post('/settings/reset', resetSettings);

/**
 * Loyiha endpointlar
 */
router.get('/projects', getProjects);
router.get('/projects/:id', getProjectById);
router.post('/projects', validateRequestBody, validateBody(), saveProjectHandler);
router.delete('/projects/:id', deleteProjectHandler);
router.get('/projects/:id/export', exportLimiter, exportProject);

/**
 * Platform export
 */
router.get('/export/platform', exportLimiter, exportPlatform);

/**
 * Chat tarixi endpointlar
 */
router.post('/chat/save', validateRequestBody, validateBody(), saveChatHistoryHandler);
router.get('/chat/:sessionId', getChatHistoryHandler);

/**
 * Health check
 */
router.get('/health', (req, res) => {
  res.json({
    success: true,
    status: 'OK',
    timestamp: new Date().toISOString()
  });
});

export default router;
