/**
 * Input validation middleware
 * Kiruvchi ma'lumotlarni tekshirish
 */

/**
 * Request body borligini tekshirish
 */
export function validateRequestBody(req, res, next) {
  if (!req.body || Object.keys(req.body).length === 0) {
    return res.status(400).json({
      success: false,
      error: 'So\'rov bo\'sh'
    });
  }
  next();
}

/**
 * JSON formatini tekshirish
 */
export function validateJSON(err, req, res, next) {
  if (err instanceof SyntaxError && err.status === 400 && 'body' in err) {
    return res.status(400).json({
      success: false,
      error: 'Noto\'g\'ri JSON format'
    });
  }
  next();
}

/**
 * String parametrini tekshirish
 */
export function sanitizeString(str, maxLength = 10000) {
  if (typeof str !== 'string') {
    return '';
  }

  // Maksimal uzunlikni tekshirish
  if (str.length > maxLength) {
    return str.substring(0, maxLength);
  }

  return str.trim();
}

/**
 * Xavfli script taglarni olib tashlash
 */
export function removeScriptTags(html) {
  if (typeof html !== 'string') {
    return '';
  }

  // Script taglarni olib tashlash (oddiy sanitization)
  // Production da DOMPurify yoki boshqa kutubxona ishlatish yaxshiroq
  return html.replace(/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi, '');
}

/**
 * Object depth ni cheklash (JSON bomb himoyasi)
 */
export function validateObjectDepth(obj, maxDepth = 10, currentDepth = 0) {
  if (currentDepth > maxDepth) {
    throw new Error('Object juda chuqur (JSON bomb hujumidan himoya)');
  }

  if (obj && typeof obj === 'object') {
    for (const key in obj) {
      if (obj.hasOwnProperty(key)) {
        validateObjectDepth(obj[key], maxDepth, currentDepth + 1);
      }
    }
  }

  return true;
}

/**
 * Request body ni validatsiya qilish
 */
export function validateBody(maxDepth = 10) {
  return (req, res, next) => {
    try {
      if (req.body) {
        validateObjectDepth(req.body, maxDepth);
      }
      next();
    } catch (error) {
      return res.status(400).json({
        success: false,
        error: error.message
      });
    }
  };
}
