import { getSettings, saveSettings } from '../utils/storage.js';
import { validateApiKey } from '../utils/encryption.js';

/**
 * Sozlamalar bilan bog'liq API endpointlar
 */

/**
 * Sozlamalarni olish
 */
export async function getSettingsHandler(req, res) {
  try {
    const settings = getSettings();

    // API keylarni frontendga yubormasdan, faqat mavjudligini ko'rsatish
    const sanitizedSettings = {
      ...settings,
      apiKeys: {}
    };

    // Har bir provider uchun faqat key mavjudligini ko'rsatish
    for (const [provider, key] of Object.entries(settings.apiKeys)) {
      sanitizedSettings.apiKeys[provider] = key ? '********' : '';
    }

    res.json({
      success: true,
      settings: sanitizedSettings
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Sozlamalarni saqlash
 */
export async function saveSettingsHandler(req, res) {
  try {
    const newSettings = req.body;

    if (!newSettings) {
      return res.status(400).json({
        success: false,
        error: 'Sozlamalar yuborilmagan'
      });
    }

    // Mavjud sozlamalarni olish
    const currentSettings = getSettings();

    // Yangi sozlamalarni merge qilish
    const updatedSettings = {
      ...currentSettings,
      ...newSettings
    };

    // API keylarni validatsiya qilish
    if (newSettings.apiKeys) {
      for (const [provider, key] of Object.entries(newSettings.apiKeys)) {
        // Agar yangi key '********' bo'lsa, eski keyni saqlab qolish
        if (key === '********') {
          updatedSettings.apiKeys[provider] = currentSettings.apiKeys[provider];
          continue;
        }

        // Bo'sh key bo'lsa, o'chirish
        if (!key || key.trim() === '') {
          updatedSettings.apiKeys[provider] = '';
          continue;
        }

        // Validatsiya
        if (!validateApiKey(key, provider)) {
          return res.status(400).json({
            success: false,
            error: `${provider} uchun noto'g'ri API key formati`
          });
        }

        updatedSettings.apiKeys[provider] = key;
      }
    }

    // Saqlash
    const result = saveSettings(updatedSettings);

    if (result.success) {
      res.json({
        success: true,
        message: 'Sozlamalar muvaffaqiyatli saqlandi'
      });
    } else {
      res.status(500).json(result);
    }
  } catch (error) {
    console.error('Sozlamalarni saqlashda xato:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Bitta API keyni yangilash
 */
export async function updateApiKey(req, res) {
  try {
    const { provider, apiKey } = req.body;

    if (!provider) {
      return res.status(400).json({
        success: false,
        error: 'Provider ko\'rsatilmagan'
      });
    }

    // Validatsiya
    if (apiKey && !validateApiKey(apiKey, provider)) {
      return res.status(400).json({
        success: false,
        error: `Noto'g'ri API key formati`
      });
    }

    // Mavjud sozlamalarni olish
    const settings = getSettings();

    // API keyni yangilash
    settings.apiKeys[provider] = apiKey || '';

    // Saqlash
    const result = saveSettings(settings);

    if (result.success) {
      res.json({
        success: true,
        message: 'API key muvaffaqiyatli yangilandi'
      });
    } else {
      res.status(500).json(result);
    }
  } catch (error) {
    console.error('API keyni yangilashda xato:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Standart sozlamalarni qaytarish
 */
export async function resetSettings(req, res) {
  try {
    const defaultSettings = {
      apiKeys: {},
      defaultProvider: 'openai',
      defaultModel: null,
      temperature: 0.7,
      maxTokens: 2000
    };

    const result = saveSettings(defaultSettings);

    if (result.success) {
      res.json({
        success: true,
        message: 'Sozlamalar standart holatga qaytarildi'
      });
    } else {
      res.status(500).json(result);
    }
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}
