import {
  saveProject,
  getProject,
  getAllProjects,
  deleteProject
} from '../utils/storage.js';
import archiver from 'archiver';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

/**
 * Loyiha bilan bog'liq API endpointlar
 */

/**
 * Barcha loyihalar ro'yxati
 */
export async function getProjects(req, res) {
  try {
    const projects = getAllProjects();
    res.json({
      success: true,
      projects
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Bitta loyiha ma'lumotlarini olish
 */
export async function getProjectById(req, res) {
  try {
    const { id } = req.params;

    if (!id) {
      return res.status(400).json({
        success: false,
        error: 'Loyiha ID ko\'rsatilmagan'
      });
    }

    const project = getProject(id);

    if (!project) {
      return res.status(404).json({
        success: false,
        error: 'Loyiha topilmadi'
      });
    }

    res.json({
      success: true,
      project
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Loyihani saqlash yoki yangilash
 */
export async function saveProjectHandler(req, res) {
  try {
    const { id, name, files, settings } = req.body;

    if (!id) {
      return res.status(400).json({
        success: false,
        error: 'Loyiha ID ko\'rsatilmagan'
      });
    }

    // Mavjud loyihani olish yoki yangi yaratish
    const existingProject = getProject(id) || {};

    const projectData = {
      id,
      name: name || existingProject.name || 'Yangi Loyiha',
      files: files || existingProject.files || {
        'index.html': '<h1>Salom Dunyo</h1>',
        'style.css': 'body { margin: 0; padding: 20px; }',
        'script.js': 'console.log("Ishga tushdi");'
      },
      settings: settings || existingProject.settings || {},
      createdAt: existingProject.createdAt || new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };

    const result = saveProject(id, projectData);

    if (result.success) {
      res.json({
        success: true,
        message: 'Loyiha muvaffaqiyatli saqlandi',
        project: projectData
      });
    } else {
      res.status(500).json(result);
    }
  } catch (error) {
    console.error('Loyihani saqlashda xato:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Loyihani o'chirish
 */
export async function deleteProjectHandler(req, res) {
  try {
    const { id } = req.params;

    if (!id) {
      return res.status(400).json({
        success: false,
        error: 'Loyiha ID ko\'rsatilmagan'
      });
    }

    const result = deleteProject(id);

    if (result.success) {
      res.json({
        success: true,
        message: 'Loyiha muvaffaqiyatli o\'chirildi'
      });
    } else {
      res.status(404).json(result);
    }
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Loyihani ZIP qilib yuklab olish
 */
export async function exportProject(req, res) {
  try {
    const { id } = req.params;

    if (!id) {
      return res.status(400).json({
        success: false,
        error: 'Loyiha ID ko\'rsatilmagan'
      });
    }

    const project = getProject(id);

    if (!project) {
      return res.status(404).json({
        success: false,
        error: 'Loyiha topilmadi'
      });
    }

    // ZIP yaratish
    const archive = archiver('zip', {
      zlib: { level: 9 }
    });

    // Response headers
    res.attachment(`${project.name || id}.zip`);
    res.setHeader('Content-Type', 'application/zip');

    // Archive ni response ga pipe qilish
    archive.pipe(res);

    // Loyiha fayllarini qo'shish
    if (project.files) {
      for (const [filename, content] of Object.entries(project.files)) {
        archive.append(content, { name: filename });
      }
    }

    // README.md yaratish
    const readme = `# ${project.name || id}

Loyiha AI Builder Platform da yaratilgan.

## Ishga tushirish

1. Fayllarni brauzerda oching
2. \`index.html\` ni oching

## Yaratilgan sana

${new Date(project.createdAt).toLocaleString('uz-UZ')}

## Oxirgi yangilanish

${new Date(project.updatedAt).toLocaleString('uz-UZ')}
`;

    archive.append(readme, { name: 'README.md' });

    // Archive ni finalize qilish
    archive.finalize();

    archive.on('error', (err) => {
      console.error('Archive xatosi:', err);
      res.status(500).json({
        success: false,
        error: 'ZIP yaratishda xato'
      });
    });
  } catch (error) {
    console.error('Export xatosi:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Butun platformani ZIP qilib yuklab olish
 */
export async function exportPlatform(req, res) {
  try {
    const archive = archiver('zip', {
      zlib: { level: 9 }
    });

    res.attachment('ai-builder-platform.zip');
    res.setHeader('Content-Type', 'application/zip');

    archive.pipe(res);

    // Backend fayllarini qo'shish
    const backendDir = path.join(__dirname, '..');
    archive.directory(backendDir, 'backend', (entry) => {
      // node_modules va storage ni exclude qilish
      if (entry.name.includes('node_modules') || entry.name.includes('storage')) {
        return false;
      }
      return entry;
    });

    // Frontend fayllarini qo'shish
    const frontendDir = path.join(__dirname, '..', '..', 'frontend');
    archive.directory(frontendDir, 'frontend', (entry) => {
      // node_modules, build, dist ni exclude qilish
      if (entry.name.includes('node_modules') ||
          entry.name.includes('build') ||
          entry.name.includes('dist')) {
        return false;
      }
      return entry;
    });

    archive.finalize();

    archive.on('error', (err) => {
      console.error('Platform export xatosi:', err);
      res.status(500).json({
        success: false,
        error: 'Platform ni export qilishda xato'
      });
    });
  } catch (error) {
    console.error('Platform export xatosi:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}
