import { aiOrchestrator } from '../services/AIOrchestrator.js';
import { getSettings } from '../utils/storage.js';

/**
 * AI bilan bog'liq barcha API endpointlar
 */

/**
 * Qo'llab-quvvatlanadigan AI providerlar ro'yxati
 */
export async function getSupportedProviders(req, res) {
  try {
    const providers = aiOrchestrator.getSupportedProviders();
    res.json({
      success: true,
      providers
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * API connection test qilish
 */
export async function testApiConnection(req, res) {
  try {
    const { provider, apiKey, config } = req.body;

    if (!provider) {
      return res.status(400).json({
        success: false,
        error: 'Provider ko\'rsatilmagan'
      });
    }

    // Ollama bundan mustasno, barcha providerlar uchun API key kerak
    if (provider !== 'ollama' && !apiKey) {
      return res.status(400).json({
        success: false,
        error: 'API key ko\'rsatilmagan'
      });
    }

    const result = await aiOrchestrator.testConnection(provider, apiKey, config || {});
    res.json(result);
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * Mavjud modellar ro'yxati
 */
export async function getModels(req, res) {
  try {
    const { provider, apiKey, config } = req.body;

    if (!provider) {
      return res.status(400).json({
        success: false,
        error: 'Provider ko\'rsatilmagan'
      });
    }

    let keyToUse = apiKey;

    // Agar API key berilmagan bo'lsa, settings dan olish
    if (!keyToUse && provider !== 'ollama') {
      const settings = getSettings();
      keyToUse = settings.apiKeys[provider];

      if (!keyToUse) {
        return res.status(400).json({
          success: false,
          error: 'API key topilmadi. Sozlamalarda API key kiriting.'
        });
      }
    }

    const result = await aiOrchestrator.getModels(provider, keyToUse, config || {});
    res.json(result);
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * AI chat (oddiy, streaming emas)
 */
export async function chat(req, res) {
  try {
    const { provider, messages, options } = req.body;

    if (!provider) {
      return res.status(400).json({
        success: false,
        error: 'Provider ko\'rsatilmagan'
      });
    }

    if (!messages || !Array.isArray(messages) || messages.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'Xabarlar massivi bo\'sh'
      });
    }

    // API key ni settings dan olish
    const settings = getSettings();
    const apiKey = settings.apiKeys[provider];

    if (!apiKey && provider !== 'ollama') {
      return res.status(400).json({
        success: false,
        error: 'API key topilmadi. Sozlamalarda API key kiriting.'
      });
    }

    const result = await aiOrchestrator.chat(provider, apiKey, messages, options || {});
    res.json(result);
  } catch (error) {
    console.error('Chat error:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}

/**
 * AI chat streaming
 */
export async function chatStream(req, res) {
  try {
    const { provider, messages, options } = req.body;

    if (!provider) {
      return res.status(400).json({
        success: false,
        error: 'Provider ko\'rsatilmagan'
      });
    }

    if (!messages || !Array.isArray(messages) || messages.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'Xabarlar massivi bo\'sh'
      });
    }

    // API key ni settings dan olish
    const settings = getSettings();
    const apiKey = settings.apiKeys[provider];

    if (!apiKey && provider !== 'ollama') {
      return res.status(400).json({
        success: false,
        error: 'API key topilmadi. Sozlamalarda API key kiriting.'
      });
    }

    // SSE headers
    res.setHeader('Content-Type', 'text/event-stream');
    res.setHeader('Cache-Control', 'no-cache');
    res.setHeader('Connection', 'keep-alive');

    // Chunk callback
    const onChunk = (content) => {
      res.write(`data: ${JSON.stringify({ content })}\n\n`);
    };

    try {
      const result = await aiOrchestrator.chatStream(
        provider,
        apiKey,
        messages,
        options || {},
        onChunk
      );

      // Tugash signali
      res.write(`data: ${JSON.stringify({ done: true, success: result.success })}\n\n`);
      res.end();
    } catch (streamError) {
      res.write(`data: ${JSON.stringify({ error: streamError.message, done: true })}\n\n`);
      res.end();
    }
  } catch (error) {
    console.error('Chat stream error:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
}
