import axios from 'axios';
import { BaseAdapter } from './BaseAdapter.js';

/**
 * OpenAI Provider Adapter
 */
export class OpenAIAdapter extends BaseAdapter {
  constructor(apiKey, config = {}) {
    super(apiKey, config);
    this.baseURL = config.baseURL || 'https://api.openai.com/v1';
    this.defaultModel = config.defaultModel || 'gpt-4-turbo-preview';
  }

  getProviderName() {
    return 'OpenAI';
  }

  /**
   * API ulanishini tekshirish
   */
  async testConnection() {
    try {
      const response = await axios.get(`${this.baseURL}/models`, {
        headers: {
          'Authorization': `Bearer ${this.apiKey}`,
          'Content-Type': 'application/json'
        },
        timeout: 10000
      });

      return {
        success: true,
        message: 'OpenAI API muvaffaqiyatli ulandi',
        models: response.data.data.slice(0, 5).map(m => m.id)
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Mavjud modellar ro'yxati
   */
  async getModels() {
    try {
      const response = await axios.get(`${this.baseURL}/models`, {
        headers: {
          'Authorization': `Bearer ${this.apiKey}`
        }
      });

      const models = response.data.data
        .filter(m => m.id.includes('gpt'))
        .map(m => ({
          id: m.id,
          name: m.id,
          provider: 'openai'
        }));

      return { success: true, models };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Chat completion (oddiy)
   */
  async chat(messages, options = {}) {
    try {
      const sanitizedMessages = messages.map(msg => ({
        role: msg.role,
        content: this.sanitizePrompt(msg.content)
      }));

      const response = await axios.post(
        `${this.baseURL}/chat/completions`,
        {
          model: options.model || this.defaultModel,
          messages: sanitizedMessages,
          temperature: options.temperature || 0.7,
          max_tokens: options.maxTokens || 2000,
          stream: false
        },
        {
          headers: {
            'Authorization': `Bearer ${this.apiKey}`,
            'Content-Type': 'application/json'
          },
          timeout: 60000
        }
      );

      return {
        success: true,
        content: response.data.choices[0].message.content,
        usage: response.data.usage,
        model: response.data.model
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Chat streaming
   */
  async chatStream(messages, options = {}, onChunk) {
    try {
      const sanitizedMessages = messages.map(msg => ({
        role: msg.role,
        content: this.sanitizePrompt(msg.content)
      }));

      const response = await axios.post(
        `${this.baseURL}/chat/completions`,
        {
          model: options.model || this.defaultModel,
          messages: sanitizedMessages,
          temperature: options.temperature || 0.7,
          max_tokens: options.maxTokens || 2000,
          stream: true
        },
        {
          headers: {
            'Authorization': `Bearer ${this.apiKey}`,
            'Content-Type': 'application/json'
          },
          responseType: 'stream',
          timeout: 120000
        }
      );

      return new Promise((resolve, reject) => {
        let fullContent = '';

        response.data.on('data', (chunk) => {
          const lines = chunk.toString().split('\n').filter(line => line.trim() !== '');

          for (const line of lines) {
            if (line.includes('[DONE]')) {
              resolve({
                success: true,
                content: fullContent
              });
              return;
            }

            if (line.startsWith('data: ')) {
              try {
                const json = JSON.parse(line.slice(6));
                const content = json.choices[0]?.delta?.content;

                if (content) {
                  fullContent += content;
                  if (onChunk) {
                    onChunk(content);
                  }
                }
              } catch (e) {
                // JSON parse xatosi - ignore
              }
            }
          }
        });

        response.data.on('end', () => {
          resolve({
            success: true,
            content: fullContent
          });
        });

        response.data.on('error', (error) => {
          reject(error);
        });
      });
    } catch (error) {
      return this.formatError(error);
    }
  }
}
