import axios from 'axios';
import { BaseAdapter } from './BaseAdapter.js';

/**
 * Ollama (Local LLM) Provider Adapter
 */
export class OllamaAdapter extends BaseAdapter {
  constructor(apiKey, config = {}) {
    // Ollama API key talab qilmaydi, lekin base class uchun dummy key beramiz
    super(apiKey || 'ollama-local', config);
    this.baseURL = config.baseURL || 'http://localhost:11434';
    this.defaultModel = config.defaultModel || 'llama2';
  }

  getProviderName() {
    return 'Ollama (Local)';
  }

  /**
   * API ulanishini tekshirish
   */
  async testConnection() {
    try {
      const response = await axios.get(`${this.baseURL}/api/tags`, {
        timeout: 5000
      });

      const models = response.data.models || [];

      return {
        success: true,
        message: 'Ollama muvaffaqiyatli ulandi',
        models: models.slice(0, 5).map(m => m.name)
      };
    } catch (error) {
      if (error.code === 'ECONNREFUSED') {
        return {
          success: false,
          error: 'Ollama ishga tushirilmagan. Iltimos, Ollama dasturini ishga tushiring.'
        };
      }
      return this.formatError(error);
    }
  }

  /**
   * Mavjud modellar ro'yxati
   */
  async getModels() {
    try {
      const response = await axios.get(`${this.baseURL}/api/tags`, {
        timeout: 5000
      });

      const models = (response.data.models || []).map(m => ({
        id: m.name,
        name: m.name,
        provider: 'ollama',
        size: m.size
      }));

      return { success: true, models };
    } catch (error) {
      if (error.code === 'ECONNREFUSED') {
        return {
          success: false,
          error: 'Ollama ishga tushirilmagan',
          models: []
        };
      }
      return this.formatError(error);
    }
  }

  /**
   * Xabarlarni Ollama formatiga keltirish
   */
  formatMessages(messages) {
    let systemPrompt = '';
    let userMessages = [];

    for (const msg of messages) {
      if (msg.role === 'system') {
        systemPrompt = this.sanitizePrompt(msg.content);
      } else {
        userMessages.push({
          role: msg.role,
          content: this.sanitizePrompt(msg.content)
        });
      }
    }

    return { systemPrompt, userMessages };
  }

  /**
   * Chat completion (oddiy)
   */
  async chat(messages, options = {}) {
    try {
      const { systemPrompt, userMessages } = this.formatMessages(messages);

      const requestBody = {
        model: options.model || this.defaultModel,
        messages: userMessages,
        stream: false,
        options: {
          temperature: options.temperature || 0.7,
          num_predict: options.maxTokens || 2000
        }
      };

      if (systemPrompt) {
        requestBody.system = systemPrompt;
      }

      const response = await axios.post(
        `${this.baseURL}/api/chat`,
        requestBody,
        {
          timeout: 120000
        }
      );

      return {
        success: true,
        content: response.data.message.content,
        model: response.data.model,
        usage: {
          prompt_tokens: response.data.prompt_eval_count,
          completion_tokens: response.data.eval_count,
          total_tokens: (response.data.prompt_eval_count || 0) + (response.data.eval_count || 0)
        }
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Chat streaming
   */
  async chatStream(messages, options = {}, onChunk) {
    try {
      const { systemPrompt, userMessages } = this.formatMessages(messages);

      const requestBody = {
        model: options.model || this.defaultModel,
        messages: userMessages,
        stream: true,
        options: {
          temperature: options.temperature || 0.7,
          num_predict: options.maxTokens || 2000
        }
      };

      if (systemPrompt) {
        requestBody.system = systemPrompt;
      }

      const response = await axios.post(
        `${this.baseURL}/api/chat`,
        requestBody,
        {
          responseType: 'stream',
          timeout: 120000
        }
      );

      return new Promise((resolve, reject) => {
        let fullContent = '';

        response.data.on('data', (chunk) => {
          const lines = chunk.toString().split('\n').filter(line => line.trim() !== '');

          for (const line of lines) {
            try {
              const json = JSON.parse(line);
              const content = json.message?.content;

              if (content) {
                fullContent += content;
                if (onChunk) {
                  onChunk(content);
                }
              }

              if (json.done) {
                resolve({
                  success: true,
                  content: fullContent
                });
              }
            } catch (e) {
              // JSON parse xatosi - ignore
            }
          }
        });

        response.data.on('end', () => {
          resolve({
            success: true,
            content: fullContent
          });
        });

        response.data.on('error', (error) => {
          reject(error);
        });
      });
    } catch (error) {
      return this.formatError(error);
    }
  }
}
