import axios from 'axios';
import { BaseAdapter } from './BaseAdapter.js';

/**
 * Google Gemini Provider Adapter
 */
export class GoogleAdapter extends BaseAdapter {
  constructor(apiKey, config = {}) {
    super(apiKey, config);
    this.baseURL = config.baseURL || 'https://generativelanguage.googleapis.com/v1beta';
    this.defaultModel = config.defaultModel || 'gemini-pro';
  }

  getProviderName() {
    return 'Google Gemini';
  }

  /**
   * API ulanishini tekshirish
   */
  async testConnection() {
    try {
      const response = await axios.post(
        `${this.baseURL}/models/${this.defaultModel}:generateContent?key=${this.apiKey}`,
        {
          contents: [{
            parts: [{ text: 'test' }]
          }]
        },
        {
          headers: {
            'Content-Type': 'application/json'
          },
          timeout: 10000
        }
      );

      return {
        success: true,
        message: 'Google Gemini API muvaffaqiyatli ulandi',
        model: this.defaultModel
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Mavjud modellar ro'yxati
   */
  async getModels() {
    const models = [
      { id: 'gemini-pro', name: 'Gemini Pro', provider: 'google' },
      { id: 'gemini-pro-vision', name: 'Gemini Pro Vision', provider: 'google' },
      { id: 'gemini-ultra', name: 'Gemini Ultra', provider: 'google' }
    ];

    return { success: true, models };
  }

  /**
   * Xabarlarni Google formatiga keltirish
   */
  formatMessages(messages) {
    const contents = [];
    let systemInstruction = '';

    for (const msg of messages) {
      if (msg.role === 'system') {
        systemInstruction = this.sanitizePrompt(msg.content);
      } else {
        contents.push({
          role: msg.role === 'assistant' ? 'model' : 'user',
          parts: [{ text: this.sanitizePrompt(msg.content) }]
        });
      }
    }

    return { systemInstruction, contents };
  }

  /**
   * Chat completion (oddiy)
   */
  async chat(messages, options = {}) {
    try {
      const { systemInstruction, contents } = this.formatMessages(messages);

      const requestBody = {
        contents: contents,
        generationConfig: {
          temperature: options.temperature || 0.7,
          maxOutputTokens: options.maxTokens || 2000,
        }
      };

      if (systemInstruction) {
        requestBody.systemInstruction = {
          parts: [{ text: systemInstruction }]
        };
      }

      const model = options.model || this.defaultModel;
      const response = await axios.post(
        `${this.baseURL}/models/${model}:generateContent?key=${this.apiKey}`,
        requestBody,
        {
          headers: {
            'Content-Type': 'application/json'
          },
          timeout: 60000
        }
      );

      const content = response.data.candidates[0].content.parts[0].text;

      return {
        success: true,
        content: content,
        usage: response.data.usageMetadata,
        model: model
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Chat streaming
   */
  async chatStream(messages, options = {}, onChunk) {
    try {
      const { systemInstruction, contents } = this.formatMessages(messages);

      const requestBody = {
        contents: contents,
        generationConfig: {
          temperature: options.temperature || 0.7,
          maxOutputTokens: options.maxTokens || 2000,
        }
      };

      if (systemInstruction) {
        requestBody.systemInstruction = {
          parts: [{ text: systemInstruction }]
        };
      }

      const model = options.model || this.defaultModel;
      const response = await axios.post(
        `${this.baseURL}/models/${model}:streamGenerateContent?key=${this.apiKey}&alt=sse`,
        requestBody,
        {
          headers: {
            'Content-Type': 'application/json'
          },
          responseType: 'stream',
          timeout: 120000
        }
      );

      return new Promise((resolve, reject) => {
        let fullContent = '';

        response.data.on('data', (chunk) => {
          const lines = chunk.toString().split('\n').filter(line => line.trim() !== '');

          for (const line of lines) {
            if (line.startsWith('data: ')) {
              try {
                const json = JSON.parse(line.slice(6));
                const content = json.candidates?.[0]?.content?.parts?.[0]?.text;

                if (content) {
                  fullContent += content;
                  if (onChunk) {
                    onChunk(content);
                  }
                }
              } catch (e) {
                // JSON parse xatosi - ignore
              }
            }
          }
        });

        response.data.on('end', () => {
          resolve({
            success: true,
            content: fullContent
          });
        });

        response.data.on('error', (error) => {
          reject(error);
        });
      });
    } catch (error) {
      return this.formatError(error);
    }
  }
}
