import axios from 'axios';
import { BaseAdapter } from './BaseAdapter.js';

/**
 * DeepSeek Provider Adapter
 * OpenAI-compatible API
 */
export class DeepSeekAdapter extends BaseAdapter {
  constructor(apiKey, config = {}) {
    super(apiKey, config);
    this.baseURL = config.baseURL || 'https://api.deepseek.com/v1';
    this.defaultModel = config.defaultModel || 'deepseek-chat';
  }

  getProviderName() {
    return 'DeepSeek';
  }

  /**
   * API ulanishini tekshirish
   */
  async testConnection() {
    try {
      const response = await axios.post(
        `${this.baseURL}/chat/completions`,
        {
          model: this.defaultModel,
          messages: [{ role: 'user', content: 'test' }],
          max_tokens: 10
        },
        {
          headers: {
            'Authorization': `Bearer ${this.apiKey}`,
            'Content-Type': 'application/json'
          },
          timeout: 10000
        }
      );

      return {
        success: true,
        message: 'DeepSeek API muvaffaqiyatli ulandi',
        model: response.data.model
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Mavjud modellar ro'yxati
   */
  async getModels() {
    const models = [
      { id: 'deepseek-chat', name: 'DeepSeek Chat', provider: 'deepseek' },
      { id: 'deepseek-coder', name: 'DeepSeek Coder', provider: 'deepseek' }
    ];

    return { success: true, models };
  }

  /**
   * Chat completion (oddiy)
   */
  async chat(messages, options = {}) {
    try {
      const sanitizedMessages = messages.map(msg => ({
        role: msg.role,
        content: this.sanitizePrompt(msg.content)
      }));

      const response = await axios.post(
        `${this.baseURL}/chat/completions`,
        {
          model: options.model || this.defaultModel,
          messages: sanitizedMessages,
          temperature: options.temperature || 0.7,
          max_tokens: options.maxTokens || 2000
        },
        {
          headers: {
            'Authorization': `Bearer ${this.apiKey}`,
            'Content-Type': 'application/json'
          },
          timeout: 60000
        }
      );

      return {
        success: true,
        content: response.data.choices[0].message.content,
        usage: response.data.usage,
        model: response.data.model
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Chat streaming
   */
  async chatStream(messages, options = {}, onChunk) {
    try {
      const sanitizedMessages = messages.map(msg => ({
        role: msg.role,
        content: this.sanitizePrompt(msg.content)
      }));

      const response = await axios.post(
        `${this.baseURL}/chat/completions`,
        {
          model: options.model || this.defaultModel,
          messages: sanitizedMessages,
          temperature: options.temperature || 0.7,
          max_tokens: options.maxTokens || 2000,
          stream: true
        },
        {
          headers: {
            'Authorization': `Bearer ${this.apiKey}`,
            'Content-Type': 'application/json'
          },
          responseType: 'stream',
          timeout: 120000
        }
      );

      return new Promise((resolve, reject) => {
        let fullContent = '';

        response.data.on('data', (chunk) => {
          const lines = chunk.toString().split('\n').filter(line => line.trim() !== '');

          for (const line of lines) {
            if (line.includes('[DONE]')) {
              resolve({
                success: true,
                content: fullContent
              });
              return;
            }

            if (line.startsWith('data: ')) {
              try {
                const json = JSON.parse(line.slice(6));
                const content = json.choices[0]?.delta?.content;

                if (content) {
                  fullContent += content;
                  if (onChunk) {
                    onChunk(content);
                  }
                }
              } catch (e) {
                // JSON parse xatosi - ignore
              }
            }
          }
        });

        response.data.on('end', () => {
          resolve({
            success: true,
            content: fullContent
          });
        });

        response.data.on('error', (error) => {
          reject(error);
        });
      });
    } catch (error) {
      return this.formatError(error);
    }
  }
}
