import axios from 'axios';
import { BaseAdapter } from './BaseAdapter.js';

/**
 * Anthropic (Claude) Provider Adapter
 */
export class AnthropicAdapter extends BaseAdapter {
  constructor(apiKey, config = {}) {
    super(apiKey, config);
    this.baseURL = config.baseURL || 'https://api.anthropic.com/v1';
    this.defaultModel = config.defaultModel || 'claude-3-sonnet-20240229';
    this.apiVersion = '2023-06-01';
  }

  getProviderName() {
    return 'Anthropic';
  }

  /**
   * API ulanishini tekshirish
   */
  async testConnection() {
    try {
      // Test uchun qisqa xabar yuboramiz
      const response = await axios.post(
        `${this.baseURL}/messages`,
        {
          model: this.defaultModel,
          max_tokens: 10,
          messages: [{ role: 'user', content: 'test' }]
        },
        {
          headers: {
            'x-api-key': this.apiKey,
            'anthropic-version': this.apiVersion,
            'Content-Type': 'application/json'
          },
          timeout: 10000
        }
      );

      return {
        success: true,
        message: 'Anthropic API muvaffaqiyatli ulandi',
        model: response.data.model
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Mavjud modellar ro'yxati
   */
  async getModels() {
    // Anthropic API modellar listini bermaydi, shuning uchun qo'lda yozamiz
    const models = [
      { id: 'claude-3-opus-20240229', name: 'Claude 3 Opus', provider: 'anthropic' },
      { id: 'claude-3-sonnet-20240229', name: 'Claude 3 Sonnet', provider: 'anthropic' },
      { id: 'claude-3-haiku-20240307', name: 'Claude 3 Haiku', provider: 'anthropic' },
      { id: 'claude-2.1', name: 'Claude 2.1', provider: 'anthropic' }
    ];

    return { success: true, models };
  }

  /**
   * Xabarlarni Anthropic formatiga keltirish
   */
  formatMessages(messages) {
    const formatted = [];
    let systemPrompt = '';

    for (const msg of messages) {
      if (msg.role === 'system') {
        systemPrompt = this.sanitizePrompt(msg.content);
      } else {
        formatted.push({
          role: msg.role,
          content: this.sanitizePrompt(msg.content)
        });
      }
    }

    return { system: systemPrompt, messages: formatted };
  }

  /**
   * Chat completion (oddiy)
   */
  async chat(messages, options = {}) {
    try {
      const { system, messages: formattedMessages } = this.formatMessages(messages);

      const requestBody = {
        model: options.model || this.defaultModel,
        max_tokens: options.maxTokens || 2000,
        messages: formattedMessages,
        temperature: options.temperature || 0.7
      };

      if (system) {
        requestBody.system = system;
      }

      const response = await axios.post(
        `${this.baseURL}/messages`,
        requestBody,
        {
          headers: {
            'x-api-key': this.apiKey,
            'anthropic-version': this.apiVersion,
            'Content-Type': 'application/json'
          },
          timeout: 60000
        }
      );

      return {
        success: true,
        content: response.data.content[0].text,
        usage: response.data.usage,
        model: response.data.model
      };
    } catch (error) {
      return this.formatError(error);
    }
  }

  /**
   * Chat streaming
   */
  async chatStream(messages, options = {}, onChunk) {
    try {
      const { system, messages: formattedMessages } = this.formatMessages(messages);

      const requestBody = {
        model: options.model || this.defaultModel,
        max_tokens: options.maxTokens || 2000,
        messages: formattedMessages,
        temperature: options.temperature || 0.7,
        stream: true
      };

      if (system) {
        requestBody.system = system;
      }

      const response = await axios.post(
        `${this.baseURL}/messages`,
        requestBody,
        {
          headers: {
            'x-api-key': this.apiKey,
            'anthropic-version': this.apiVersion,
            'Content-Type': 'application/json'
          },
          responseType: 'stream',
          timeout: 120000
        }
      );

      return new Promise((resolve, reject) => {
        let fullContent = '';

        response.data.on('data', (chunk) => {
          const lines = chunk.toString().split('\n').filter(line => line.trim() !== '');

          for (const line of lines) {
            if (line.startsWith('data: ')) {
              try {
                const json = JSON.parse(line.slice(6));

                if (json.type === 'content_block_delta') {
                  const content = json.delta?.text;
                  if (content) {
                    fullContent += content;
                    if (onChunk) {
                      onChunk(content);
                    }
                  }
                }

                if (json.type === 'message_stop') {
                  resolve({
                    success: true,
                    content: fullContent
                  });
                }
              } catch (e) {
                // JSON parse xatosi - ignore
              }
            }
          }
        });

        response.data.on('end', () => {
          resolve({
            success: true,
            content: fullContent
          });
        });

        response.data.on('error', (error) => {
          reject(error);
        });
      });
    } catch (error) {
      return this.formatError(error);
    }
  }
}
